package ga.view.interfaces;

import com.jme3.input.controls.ActionListener;

/**
 * Abstract class for an mouse listener. Extends {@link ActionListener} and adds
 * functions.
 * 
 * @since 12.08.2012
 * @author Stephan Dreyer
 */
public abstract class MouseListener implements ActionListener {
  private long lastTime;
  private boolean enabled = true;

  /**
   * Instantiates a new mouse listener.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public MouseListener() {
  }

  @Override
  public final void onAction(final String name, final boolean keyPressed,
      final float tpf) {
    if (enabled) {
      boolean isDoubleClick = false;
      if (keyPressed) {
        final long thisTime = System.currentTimeMillis() - lastTime;
        lastTime = System.currentTimeMillis();

        // TODO EXTERNALIZE THE CLICK TIME
        isDoubleClick = thisTime < 1000;
      }
      onAction(name, keyPressed, isDoubleClick, tpf);
    }
  }

  /**
   * Callback method for mouse events. Subclasses must implement this to add
   * custom code.
   * 
   * @param name
   *          Name of the action.
   * @param keyPressed
   *          <code>true</code> if the key is still pressed.
   * @param isDoubleClick
   *          <code>true</code> if it is a double click.
   * @param tpf
   *          The time per frame.
   * 
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public abstract void onAction(String name, boolean keyPressed,
      boolean isDoubleClick, float tpf);

  /**
   * Sets the listener enabled.
   * 
   * @param enabled
   *          the new enabled state.
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public void setEnabled(final boolean enabled) {
    this.enabled = enabled;
  }

  /**
   * Checks if is enabled.
   * 
   * @return true, if is enabled
   * @since 12.08.2012
   * @author Stephan Dreyer
   */
  public boolean isEnabled() {
    return enabled;
  }
}
